/*
	Simple space-warp/distortion vertex program demo
	(Press the space bar to switch through programs)

	sgreen@nvidia.com 9/2000, based on Cass's vtxprog_silhouette
*/

#define GLH_EXT_SINGLE_FILE
#define GLH_NVEB_USING_NVPARSE

#include <glh_nveb.h>

#include <glh_extensions.h>
#include <glh_obs.h>
#include <glh_glut.h>

#include <array_texture.h>
#include <data_path.h>
#include <nv_png.h>

#include "vertex_programs.h"
#include "unaliased_geometry.h"

#include <nvparse.h>

NVEB_EFFECT_NAME("Warp");
NVEB_EFFECT_VERSION("1.0");
NVEB_EFFECT_LOCATION("Effects\\Animation");
NVEB_EFFECT_ABOUT(0, NULL, "Warp", "NVEffectsExplained.htm#warp");
NVEB_EFFECT_ABOUT(1, "Date", "June 2001", NULL);
NVEB_EFFECT_ABOUT(2, NULL, "Developer Relations", "http://www.nvidia.com/developer");

using namespace glh;

#define M_PI		3.1415927f
#define SIN_PERIOD	3.079	// period of 4-term Taylor approximation to sin isn't quite 2*M_PI

glut_callbacks cb;
glut_perspective_reshaper reshaper(60.0, 0.1, 10.0);
glut_simple_mouse_interactor mouse;

#define NOBJS   5
#define NPROGS	7
vertex_program vtx_prog[NPROGS];

unsigned char *vprogCode[] = {
	vprog::transform_light,
	vprog::pulsate,
	vprog::wave,
	vprog::fisheye,
	vprog::spherize,
	vprog::ripple,
	vprog::twist,
};

char *vprogName[] = {
	"Normal",
	"Pulsate",
	"Wave",
	"Square fisheye",
	"Spherical fisheye",
	"Ripple",
	"Twist"
};

signed int program = 2;
unsigned int obj = 2;

bool b[256];

GLboolean wire = 0;
GLboolean animating = 1;

float anim = 0.0;
float amp = 0.05;
float freq = 8.0;
float d = 4.0;

tex_object_2D tex;

void init_opengl();

void key(unsigned char k, int x, int y);
void special(int Key, int x, int y);
void display();
void idle(void);
void setWindowTitle(void);
void draw_object(int obj);
void mainMenu(int i);

int main(int argc, char **argv)
{
	glutInit(&argc, argv);
	glutInitWindowSize(512,512);
	glutInitDisplayMode(GLUT_RGB|GLUT_DEPTH|GLUT_DOUBLE);
	glutCreateWindow("Warp");
	setWindowTitle();

	init_opengl();

#if 0
	array2<vec3ub> img;
	read_png_rgb("grand.png", img);
	tex.bind();
	make_rgb_texture(img, 1);
    tex.parameter(GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    tex.parameter(GL_TEXTURE_MIN_FILTER, GL_LINEAR);
#endif

	glut_helpers_initialize();

	cb.keyboard_function = key;
	mouse.configure_buttons(2);
	mouse.dolly.dolly[2] = -1.5;

	glut_add_interactor(&cb);
	glut_add_interactor(&reshaper);
	glut_add_interactor(&mouse);

	/*glutCreateMenu(mainMenu);
	glutAddMenuEntry("Switch vertex program [ ]", ' ');
	glutAddMenuEntry("Switch object [o]", 'o');
	glutAddMenuEntry("Toggle vertex program [p]", 'p');
	glutAddMenuEntry("Toggle wireframe [w]", 'w');
	glutAddMenuEntry("Toggle animation [a]", 'a');
	glutAddMenuEntry("Increase amplitude [=]", '=');
	glutAddMenuEntry("Decrease amplitude [-]", '-');
	glutAddMenuEntry("Increase frequency []]", ']');
	glutAddMenuEntry("Decrease frequency [[]", '[');
	glutAddMenuEntry("Reset [r]", 'r');
	glutAddMenuEntry("Quit [esc]", '\033');
	glutAttachMenu(GLUT_RIGHT_BUTTON);*/
    
    glutCreateMenu(mainMenu);
    glutAddMenuEntry("Help [h]", 'h');
	glutAttachMenu(GLUT_RIGHT_BUTTON);

	b['p'] = true;

	glutIdleFunc(idle);
	glutDisplayFunc(display);
    glutSpecialFunc(special);
	glutMainLoop();

	return 0;
}


void setWindowTitle(void)
{
	char windowTitle[256];
	sprintf(windowTitle, "SpaceWarp - %s", vprogName[program]);
	glutSetWindowTitle(windowTitle);
}

void mainMenu(int i)
{
	key((unsigned char) i, 0, 0);
}

void init_opengl()
{
	float cc = 0.0f;
	glClearColor(cc, cc, cc, 1);

	glColor3f(1,1,1);
	glEnable(GL_DEPTH_TEST);
	glDisable(GL_CULL_FACE);

    if (!glh_init_extensions("GL_NV_vertex_program")) {
		printf("GL_NV_vertex_program extension not found\n");
		exit(-1);
	}

    for(int i=0; i<NOBJS; i++) {
      glNewList(i+1, GL_COMPILE);
      draw_object(i);
      glEndList();
    }    

	for(i=0; i<NPROGS; i++) {
		vtx_prog[i].bind();
		//vtx_prog[i].load(strlen((const char *) vprogCode[i]), vprogCode[i]);
        nvparse((const char *)vprogCode[i]);
	}

	glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 0, GL_MODELVIEW_PROJECTION_NV,		GL_IDENTITY_NV);
	glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 4,               GL_MODELVIEW,		GL_INVERSE_TRANSPOSE_NV);
	glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 8,               GL_MODELVIEW,		GL_IDENTITY_NV);
	glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 12,             GL_PROJECTION,		GL_IDENTITY_NV);

	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 32, 0.0, 0.0, 1.0, 0.0);	// light position/direction
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 35, 0.0, 1.0, 0.0, 0.0);	// diffuse color
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 36, 1.0, 1.0, 1.0, 0.0);	// specular color

	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 64, 0.0, 1.0, 2.0, 3.0);	// smoothstep constants

	// sin Taylor series constants - 1, 1/3!, 1/5!, 1/7!
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 63, 1.0, 1.0 / (3*2), 1.0 / (5*4*3*2), 1.0 / (7*6*5*4*3*2));

//	vtx_prog.parameter(62, 1.0 / (2.0 * M_PI), 2.0 * M_PI, M_PI, M_PI/2.0);
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 62, 1.0 / (2.0 * SIN_PERIOD), 2.0 * SIN_PERIOD, SIN_PERIOD, SIN_PERIOD/2.0);

	// sin wave frequency, amplitude
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 61, 1.0, 0.2, 0.0, 0.0);

	// phase animation
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 60, 0.0, 0.0, 0.0, 0.0);

    // fisheye sphere radius
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 59, 1.0, 0.0, 0.0, 0.0);

    float znear = 0.01, zfar = 10.0;
    glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 58, 0.0, 0.0, -2.0 / (zfar - znear), -(zfar+znear)/(zfar-znear) );
}


void idle(void)
{
	anim -= 0.1;
	glutPostRedisplay();
}

void printProgTitle(unsigned char *s)
{
	while(*s != '\n') {
		putc(*s++, stdout);
	}
	printf("\n");
}

void special(int Key, int x, int y)
{
    switch (Key)
    {
        case GLUT_KEY_HOME:
            anim = 0.0;
            amp = 0.05;
            freq = 8.0;
            d = 4.0;
    	    mouse.dolly.dolly[2] = -1.5;
            glutPostRedisplay();
            break;
        case GLUT_KEY_LEFT:
		    program--;
            if (program < 0)
                program = NPROGS-1;
        	setWindowTitle();
            break;
            glutPostRedisplay();
        case GLUT_KEY_RIGHT:
		    program = (program + 1) % NPROGS;
        	setWindowTitle();
            glutPostRedisplay();
            break;
        case GLUT_KEY_F1:
            char *msg = "F1/h - Help\n\n"
                        "Home - Reset\n\n"
                        "Left Button & Mouse - Rotate viewpoint\n\n"
                        "1..5 - Switch object (Sphere, Torus, Triceratop, Cube, Cylinder)\n\n"
                        "- / + - Change amplitude\n\n"
                        "[ / ] - Change frequency\n\n"
                        ", / . - Change square fisheye size\n\n"
                        "Left - Next vertex program\n\n"
                        "Right - Previous vertex program\n\n";
                        "W - Toggle wireframe\n\n"
                        "Space - Toggle animation\n\n"
                        "Esc/q - Exit program\n\n";
            MessageBox(GetForegroundWindow(), msg, "Help", MB_OK | MB_ICONINFORMATION);
            break;
    }
}

void key(unsigned char k, int x, int y)
{

	b[k] = ! b[k];
	if(k==27 || k==0 || k=='q') exit(0);

	switch(k) {
	case 'w':
		wire ^= 1;
		if (wire)
			glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
		else
			glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
		break;
	case '=':
    case'+':
		amp += 0.01;
		break;
	case '-':
		amp -= 0.01;
		break;

	case ']':
		freq += 0.5;
		break;
	case '[':
		freq -= 0.5;
		break;

	case '.':
		d += 0.1;
		break;
	case ',':
		d -= 0.1;
		break;

	case 'r':
		anim = 0.0;
		amp = 0.05;
		freq = 8.0;
        d = 4.0;
	    mouse.dolly.dolly[2] = -1.5;
		break;

	case ' ':
		if (animating ^= 1)
			glutIdleFunc(idle);
		else
			glutIdleFunc(NULL);
		break;
    case '1':
        obj = 0;
        break;
    case '2':
        obj = 1;
        break;
    case '3':
        obj = 2;
        break;
    case '4':
        obj = 3;
        break;
    case '5':
        obj = 4;
        break;
    case 'h':
        special(GLUT_KEY_F1, 0, 0);
        break;
	/*case ' ':
		program = (program + 1) % NPROGS;
		setWindowTitle();
//		printProgTitle(vprogCode[program]);
		break;*/
	}

	//printf("amp: %f, freq: %f, d: %f\n", amp, freq, d);
	glutPostRedisplay();
}

void drawCylinder(void)
{
	GLUquadric *quad;

	quad = gluNewQuadric();
	gluQuadricDrawStyle(quad, GLU_FILL);
	gluQuadricOrientation(quad, GLU_OUTSIDE);
	gluQuadricNormals(quad, GLU_SMOOTH);
	gluQuadricTexture(quad, GL_TRUE);

	glMatrixMode(GL_MODELVIEW);
	glPushMatrix();
	glTranslatef(-1.0, 0.0, 0.0);
	glRotatef(90.0, 0.0, 1.0, 0.0);

	gluCylinder(quad, 0.25, 0.25, 2.0, 60, 30);
	glPopMatrix();

	gluDeleteQuadric(quad);
}

void draw_background()
{
    glDepthMask(GL_FALSE);

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();

	glMatrixMode(GL_PROJECTION);
    glPushMatrix();
	glLoadIdentity();
	glOrtho(-1.0, 1.0, -1.0, 1.0, -1.0, 1.0);

    tex.bind();
    tex.enable();
    glTexEnvi(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    draw_grid(20, 20, -10.0, -10.0, 0.0, 20.0, 0.0, 0.0, 0.0, 20.0, 0.0);
    tex.disable();

	glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();

    glDepthMask(GL_TRUE);
}

void drawCube()
{
  int cr = 40;

  glScalef(0.5, 0.5, 0.5);
  // back
  glColor3f(1.0, 0.0, 0.0);
  glNormal3f(0.0, 0.0, -1.0);
  draw_grid(cr, cr, -1.0, -1.0, -1.0, 2.0, 0.0, 0.0, 0.0, 2.0, 0.0);

  // front
  glColor3f(1.0, 0.0, 0.0);
  glNormal3f(0.0, 0.0, 1.0);
  draw_grid(cr, cr, -1.0, -1.0, 1.0, 2.0, 0.0, 0.0, 0.0, 2.0, 0.0);

  // left
  glColor3f(0.0, 1.0, 0.0);
  glNormal3f(-1.0, 0.0, 0.0);
  draw_grid(cr, cr, -1.0, -1.0, -1.0, 0.0, 0.0, 2.0, 0.0, 2.0, 0.0);

  // right
  glColor3f(0.0, 0.0, 1.0);
  glNormal3f(1.0, 0.0, 0.0);
  draw_grid(cr, cr, 1.0, -1.0, -1.0, 0.0, 0.0, 2.0, 0.0, 2.0, 0.0);

  // bottom
  glColor3f(1.0, 1.0, 0.0);
  glNormal3f(0.0,-1.0, 0.0);
  draw_grid(cr, cr, -1.0, -1.0, -1.0, 2.0, 0.0, 0.0, 0.0, 0.0, 2.0);

  // top
  glColor3f(0.0, 1.0, 1.0);
  glNormal3f(0.0, 1.0, 0.0);
  draw_grid(cr, cr, -1.0, 1.0, -1.0, 2.0, 0.0, 0.0, 0.0, 0.0, 2.0);
}

void draw_object(int obj)
{
	switch(obj) {
	case 0:
		draw_sphere(0.5f, 100, 100);
		break;

	case 1:
		draw_torus(0.25f, 0.5f, 100, 100);
		break;

	case 2:
		draw_object();
		break;

	case 3:
        drawCube();
		break;

	case 4:
		drawCylinder();
		break;
	}
}

void display()
{
	glClear(GL_COLOR_BUFFER_BIT|GL_DEPTH_BUFFER_BIT);

	glPushMatrix();
	mouse.apply_transform();

	vtx_prog[program].bind();
	glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 60, anim, 0.0, 0.0, 0.0);

    if (program==6)
      glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 61, sin(anim)*amp*50.0, 0.0, 0.0, 0.0);
    else
      glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, 61, freq, amp, d, d+1);

	if (b['p'])
		glEnable(GL_VERTEX_PROGRAM_NV);

#if 0
    if ((program>=2) && (program<=5)) 
      draw_background();
#endif
glDisable(GL_TEXTURE_2D);
    //draw_object(obj);
    glCallList(obj+1);

	glDisable(GL_VERTEX_PROGRAM_NV);

	glPopMatrix();
	glutSwapBuffers();
}
