#include "CA_Water.hpp"
#include <GL/glut.h>
#include "glh_genext.h"
#include "read_text_file.h"
#include "pbuffer.h"
#include "nv_tga.h"

#define GLH_NVEB_USING_NVPARSE

#include <glh_nveb.h>
#include <nvparse.h>

//.----------------------------------------------------------------------------.
//|   Function   : CA_Water::CA_Water                                          |
//|   Description: Constructor.                                                |
//.----------------------------------------------------------------------------.
CA_Water::CA_Water()
: 
  _iVertexProgramID(0),
  _iFlipState(0),
  _bWrap(false),
  _bReset(true),
  _bSingleStep(false),
  _bAnimate(true),
  _bSlow(true),
  _bWireframe(false),
  _bCreateNormalMap(true),
  _bApplyInteriorBoundaries(true),
  _bSpinLogo(true),
  _rPerTexelWidth(0),
  _rPerTexelHeight(0),
  _rBlurDist(0.5f),
  _rNormalSTScale(0.8f),
  _rBumpScale(0.25f),
  _rDropletFrequency(0.175f),
  _iSlowDelay(15),
  _iSkipInterval(0),
  _eRenderMode(CA_FULLSCREEN_REFLECT),
  _pPixelBuffer(NULL)
{
    memset(_pInitialMapDimensions, 0, sizeof(_pInitialMapDimensions));
    memset(_pStaticTextureIDs, 0, sizeof(_pStaticTextureIDs));
    memset(_pDynamicTextureIDs, 0, sizeof(_pDynamicTextureIDs));
    memset(_pDisplayListIDs, 0, sizeof(_pDisplayListIDs));
}


//.----------------------------------------------------------------------------.
//|   Function   : CA_Water::~CA_Water                                         |
//|   Description: Destructor                                                  |
//.----------------------------------------------------------------------------.
CA_Water::~CA_Water()
{
    for (int i = 0; i < CA_NUM_STATIC_TEXTURES; i++)
    {
        if (_pStaticTextureIDs[i])
            glDeleteTextures(1, &_pStaticTextureIDs[i]);
    }

    for (i = 0; i < CA_NUM_DYNAMIC_TEXTURES; i++)
    {
        if (_pDynamicTextureIDs[i])
            glDeleteTextures(1, &_pDynamicTextureIDs[i]);
    }
}


//.----------------------------------------------------------------------------.
//|   Function   : CA_Water::Initialize                                        |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void CA_Water::Initialize(char *pInitialMapFilename, char *pSpinFilename, char *pDropletFilename, char *pCubeMapFilename)
{
    _LoadTextures(pInitialMapFilename, pSpinFilename, pDropletFilename, pCubeMapFilename);

    // create the pbuffer.  Will use this as an offscreen rendering buffer.
    // it allows rendering a texture larger than our window.
    _pPixelBuffer = new PBuffer(_pInitialMapDimensions[0], _pInitialMapDimensions[1], GLUT_SINGLE | GLUT_RGBA );
    _pPixelBuffer->Initialize(true);
    _pPixelBuffer->MakeCurrent();

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    gluOrtho2D(-1, 1, -1, 1);	
    
    glClearColor(0, 0, 0, 0);
    glDisable(GL_LIGHTING);
    glDisable(GL_DEPTH_TEST);
    
    // make the offsets used by the vertex program to generated texcoords and 
    // load them into constant memory
    _CreateAndWriteUVOffsets(_pInitialMapDimensions[0], _pInitialMapDimensions[1]);

    char * programBuffer;

    ///////////////////////////////////////////////////////////////////////////
    // UV Offset Vertex Program
    ///////////////////////////////////////////////////////////////////////////
    // track the MVP matrix for the vertex program
    glTrackMatrixNV(GL_VERTEX_PROGRAM_NV, 0, GL_MODELVIEW_PROJECTION_NV, GL_IDENTITY_NV);
    float rCVConsts[] = { 0, 0.5f, 1.0f, 2.0f };
    glProgramParameter4fvNV(GL_VERTEX_PROGRAM_NV, CV_CONSTS_1, rCVConsts);

    glGenProgramsNV(1, &_iVertexProgramID);
    glBindProgramNV(GL_VERTEX_PROGRAM_NV, _iVertexProgramID);

    programBuffer = read_text_file("Texcoord_4_Offset.vp");
    
    nvparse(programBuffer);
  
    for (char * const * errors = nvparse_get_errors(); *errors; errors++)
        fprintf(stderr, "Vertex Program Texcoord_4_Offset.vp: %s\n", *errors);
    delete [] programBuffer;

    ///////////////////////////////////////////////////////////////////////////
    // register combiner setup for equal weight combination of texels
    ///////////////////////////////////////////////////////////////////////////
    programBuffer = read_text_file("EqWeightCombine_PostMult.rcp");
    
    _pDisplayListIDs[CA_REGCOMBINER_EQ_WEIGHT_COMBINE] = glGenLists(1);
    glNewList(_pDisplayListIDs[CA_REGCOMBINER_EQ_WEIGHT_COMBINE], GL_COMPILE);
    {
        nvparse(programBuffer);
         
        for (char * const * errors = nvparse_get_errors(); *errors; errors++)
           fprintf(stderr, "Register Combiners EqWeightCombine_PostMult.rcp error: %s\n", *errors);
        glEnable(GL_REGISTER_COMBINERS_NV);
    }
    glEndList();
    delete [] programBuffer;

    ///////////////////////////////////////////////////////////////////////////
    // register combiners setup for computing force from neighbors (step 1)
    ///////////////////////////////////////////////////////////////////////////
    programBuffer = read_text_file("NeighborForceCalcStep1.rcp");

    _pDisplayListIDs[CA_REGCOMBINER_NEIGHBOR_FORCE_CALC_1] = glGenLists(1);
    glNewList(_pDisplayListIDs[CA_REGCOMBINER_NEIGHBOR_FORCE_CALC_1], GL_COMPILE);
    {    
        nvparse(programBuffer);

        for (char * const * errors = nvparse_get_errors(); *errors; errors++)
           fprintf(stderr, "Register Combiners NeighborForceCalcStep1.rcp error: %s\n", *errors);
        glEnable(GL_REGISTER_COMBINERS_NV);
    }
    glEndList();
    delete [] programBuffer;

    ///////////////////////////////////////////////////////////////////////////
    // register combiners setup for computing force from neighbors (step 2)
    ///////////////////////////////////////////////////////////////////////////
    programBuffer = read_text_file("NeighborForceCalcStep2.rcp");

    _pDisplayListIDs[CA_REGCOMBINER_NEIGHBOR_FORCE_CALC_2] = glGenLists(1);
    glNewList(_pDisplayListIDs[CA_REGCOMBINER_NEIGHBOR_FORCE_CALC_2], GL_COMPILE);
    {    
        nvparse(programBuffer);

        for (char * const * errors = nvparse_get_errors(); *errors; errors++)
           fprintf(stderr, "Register Combiners NeighborForceCalcStep2.rcp error: %s\n", *errors);
        glEnable(GL_REGISTER_COMBINERS_NV);
    }
    glEndList();
    delete [] programBuffer;


    ///////////////////////////////////////////////////////////////////////////
    // register combiners setup to apply force
    ///////////////////////////////////////////////////////////////////////////
    programBuffer = read_text_file("ApplyForce.rcp");

    _pDisplayListIDs[CA_REGCOMBINER_APPLY_FORCE] = glGenLists(1);
    glNewList(_pDisplayListIDs[CA_REGCOMBINER_APPLY_FORCE], GL_COMPILE);
    {      
        // Configure texture shader
        nvparse(programBuffer);
            
        for (char * const * errors = nvparse_get_errors(); *errors; errors++)
            fprintf(stderr, "Register Combiners ApplyForce.rcp error: %s\n", *errors);
        glEnable(GL_REGISTER_COMBINERS_NV);
    }
    glEndList();
    delete [] programBuffer;

    ///////////////////////////////////////////////////////////////////////////
    // register combiners setup to apply velocity
    ///////////////////////////////////////////////////////////////////////////
    programBuffer = read_text_file("ApplyVelocity.rcp");

    _pDisplayListIDs[CA_REGCOMBINER_APPLY_VELOCITY] = glGenLists(1);
    glNewList(_pDisplayListIDs[CA_REGCOMBINER_APPLY_VELOCITY], GL_COMPILE);
    {      
        // Configure texture shader
        nvparse(programBuffer);
            
        for (char * const * errors = nvparse_get_errors(); *errors; errors++)
            fprintf(stderr, "Register Combiners ApplyVelocity.rcp error: %s\n", *errors);
        glEnable(GL_REGISTER_COMBINERS_NV);
    }
    glEndList();
    delete [] programBuffer;

    ///////////////////////////////////////////////////////////////////////////
    // register combiners setup to create a normal map
    ///////////////////////////////////////////////////////////////////////////
    programBuffer = read_text_file("CreateNormalMap.rcp");

    _pDisplayListIDs[CA_REGCOMBINER_CREATE_NORMAL_MAP] = glGenLists(1);
    glNewList(_pDisplayListIDs[CA_REGCOMBINER_CREATE_NORMAL_MAP], GL_COMPILE);
    {      
        // Configure texture shader
        nvparse(programBuffer);
            
        for (char * const * errors = nvparse_get_errors(); *errors; errors++)
            fprintf(stderr, "Register Combiners CreateNormalMap.rcp error: %s\n", *errors);

        glEnable(GL_REGISTER_COMBINERS_NV);
    }
    glEndList();
    delete [] programBuffer;

    ///////////////////////////////////////////////////////////////////////////
    // texture shader setup for dot product reflection
    ///////////////////////////////////////////////////////////////////////////
    programBuffer = read_text_file("DotProductReflect.ts");
    
    _pDisplayListIDs[CA_TEXTURE_SHADER_REFLECT] = glGenLists(1);
    glNewList(_pDisplayListIDs[CA_TEXTURE_SHADER_REFLECT], GL_COMPILE);
    {
        nvparse(programBuffer);
         
        for (char * const * errors = nvparse_get_errors(); *errors; errors++)
           fprintf(stderr, "Texture Shader DotProductReflect.ts error: %s\n", *errors);

       	nvparse("!!RC1.0                        \n"
	            "out.rgb = tex3;                \n"              
	            );

        glDisable(GL_BLEND);
        glEnable(GL_TEXTURE_SHADER_NV);
        glEnable(GL_REGISTER_COMBINERS_NV);
    }
    glEndList();
    delete [] programBuffer;

    ///////////////////////////////////////////////////////////////////////////
    // display list to render a single screen space quad.
    ///////////////////////////////////////////////////////////////////////////
    _pDisplayListIDs[CA_DRAW_SCREEN_QUAD] = glGenLists(1);
    glNewList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD], GL_COMPILE);
    {
        glColor4f(1, 1, 1, 1);
        glBegin(GL_TRIANGLE_STRIP);
        {
            glTexCoord2f(0, 1); glVertex2f(-1,  1);
            glTexCoord2f(0, 0); glVertex2f(-1, -1);
            glTexCoord2f(1, 1); glVertex2f( 1,  1);
            glTexCoord2f(1, 0); glVertex2f( 1, -1);
            
            
        }
        glEnd();
    }
    glEndList();
}


//.----------------------------------------------------------------------------.
//|   Function   : CA_Water::Display                                           |
//|   Description: Display the current status of the game of life.             |
//.----------------------------------------------------------------------------.
void CA_Water::Display(const matrix4f& matRotation)
{
    static int nSkip = 0;

	if( nSkip >= _iSkipInterval && (_eRenderMode != CA_DO_NOT_RENDER) )
	{
		nSkip = 0;

		// Display the results of the rendering to texture
		if( _bWireframe )
        {
            glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
           
			// chances are the texture will be all dark, so lets not use a texture
            glDisable(GL_TEXTURE_2D);
        }
        else
        {
            glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
            			
            glActiveTextureARB(GL_TEXTURE0_ARB);
            glEnable(GL_TEXTURE_2D);
        }

   		switch( _eRenderMode ) // more later
		{
        case CA_FULLSCREEN_REFLECT:
            {
                // include bump scale...
            	matrix4f bscale;
	            bscale(0,0) = _rBumpScale;
	            bscale(1,1) = _rBumpScale;
	            matrix4f matRot;
                matRot = matRotation * bscale;

                glCallList(_pDisplayListIDs[CA_TEXTURE_SHADER_REFLECT]);

                // Draw quad over full display
                glActiveTextureARB(GL_TEXTURE0_ARB);
                glBindTexture(GL_TEXTURE_2D, _pDynamicTextureIDs[CA_TEXTURE_NORMAL_MAP]);
                glDisable(GL_TEXTURE_2D);
                glActiveTextureARB(GL_TEXTURE3_ARB);
                glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, _pStaticTextureIDs[CA_TEXTURE_CUBEMAP]);
                glEnable(GL_TEXTURE_2D);

                glColor4f(1, 1, 1, 1);
                glBegin(GL_QUADS);
                
                glMultiTexCoord2fARB(GL_TEXTURE0_ARB, 0,0);
                glMultiTexCoord4fARB(GL_TEXTURE1_ARB, matRot(0,0), matRot(0,1), matRot(0,2),  1);
                glMultiTexCoord4fARB(GL_TEXTURE2_ARB, matRot(1,0), matRot(1,1), matRot(1,2),  1);
                glMultiTexCoord4fARB(GL_TEXTURE3_ARB, matRot(2,0), matRot(2,1), matRot(2,2),  1);
                glVertex2f(-1,-1);
                
                glMultiTexCoord2fARB(GL_TEXTURE0_ARB, 1,0);
                glMultiTexCoord4fARB(GL_TEXTURE1_ARB, matRot(0,0), matRot(0,1), matRot(0,2), -1);
                glMultiTexCoord4fARB(GL_TEXTURE2_ARB, matRot(1,0), matRot(1,1), matRot(1,2),  1);
                glMultiTexCoord4fARB(GL_TEXTURE3_ARB, matRot(2,0), matRot(2,1), matRot(2,2),  1);
                glVertex2f( 1,-1);
                
                glMultiTexCoord2fARB(GL_TEXTURE0_ARB, 1,1);
                glMultiTexCoord4fARB(GL_TEXTURE1_ARB, matRot(0,0), matRot(0,1), matRot(0,2), -1);
                glMultiTexCoord4fARB(GL_TEXTURE2_ARB, matRot(1,0), matRot(1,1), matRot(1,2), -1);
                glMultiTexCoord4fARB(GL_TEXTURE3_ARB, matRot(2,0), matRot(2,1), matRot(2,2),  1);
                glVertex2f( 1, 1);
                
                glMultiTexCoord2fARB(GL_TEXTURE0_ARB, 0,1);
                glMultiTexCoord4fARB(GL_TEXTURE1_ARB, matRot(0,0), matRot(0,1), matRot(0,2),  1);
                glMultiTexCoord4fARB(GL_TEXTURE2_ARB, matRot(1,0), matRot(1,1), matRot(1,2), -1);
                glMultiTexCoord4fARB(GL_TEXTURE3_ARB, matRot(2,0), matRot(2,1), matRot(2,2),  1);
                glVertex2f(-1, 1);
                
                glEnd();
    
                glDisable(GL_TEXTURE_SHADER_NV);
                glDisable(GL_REGISTER_COMBINERS_NV);
                
            }
            break;
        case CA_FULLSCREEN_NORMALMAP:
            {
                // Draw quad over full display
                glActiveTextureARB(GL_TEXTURE0_ARB);
                glBindTexture(GL_TEXTURE_2D, _pDynamicTextureIDs[CA_TEXTURE_NORMAL_MAP]);
                
                glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);
                
            }
            break;
        case CA_FULLSCREEN_HEIGHT:
            {
                // Draw quad over full display
                glActiveTextureARB(GL_TEXTURE0_ARB);
                glBindTexture(GL_TEXTURE_2D, _iTexHeightOutput);
                
                glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);
                
            }
            break;
        case CA_FULLSCREEN_FORCE:
            {
                // Draw quad over full display
                glActiveTextureARB(GL_TEXTURE0_ARB);
                glBindTexture(GL_TEXTURE_2D, _pDynamicTextureIDs[CA_TEXTURE_FORCE_TARGET]);
			                 
                glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);
			    
            }
            break;
        case CA_TILED_THREE_WINDOWS:
            {
                // Draw quad over full display
                // lower left
                glActiveTextureARB(GL_TEXTURE0_ARB);
                glBindTexture(GL_TEXTURE_2D, _pDynamicTextureIDs[CA_TEXTURE_FORCE_TARGET]);
                glMatrixMode(GL_MODELVIEW);
                glPushMatrix();
			                 
                glTranslatef(-0.5f, -0.5f, 0);
                glScalef(0.5f, 0.5f, 1);
                glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);
                glPopMatrix();

                // lower right
                glBindTexture(GL_TEXTURE_2D, _iTexVelocityOutput);
                glPushMatrix();
			                 
                glTranslatef(0.5f, -0.5f, 0);
                glScalef(0.5f, 0.5f, 1);
                glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);
                glPopMatrix();

                // upper left
                glBindTexture(GL_TEXTURE_2D, _pDynamicTextureIDs[CA_TEXTURE_NORMAL_MAP]);
                glMatrixMode(GL_MODELVIEW);
                glPushMatrix();
			                 
                glTranslatef(-0.5f, 0.5f, 0);
                glScalef(0.5f, 0.5f, 1);
                glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);
                glPopMatrix();

                // upper right
                glBindTexture(GL_TEXTURE_2D, _iTexHeightOutput);
                glMatrixMode(GL_MODELVIEW);
                glPushMatrix();
			                 
                glTranslatef(0.5f, 0.5f, 0);
                glScalef(0.5f, 0.5f, 1);
                glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);
                glPopMatrix();
			    
            }
            break;

        }
    }
   	else
	{
		// skip rendering this frame
		nSkip++;
	}   
}


//.----------------------------------------------------------------------------.
//|   Function   : CA_Water::Tick                                              |
//|   Description: Take a single step in the cellular automaton.               |
//.----------------------------------------------------------------------------.
void CA_Water::Tick()
{
    // Disable culling
	glDisable(GL_CULL_FACE);

    if(_bReset)
	{
		_bReset = false;
		_iFlipState = 0;
	}

	if(_bAnimate)
	{
		// Update the textures for one step of the simulation
        // make the pbuffer the current target
        _pPixelBuffer->MakeCurrent();

		_DoSingleTimeStep();
    }
	else if(_bSingleStep)
	{
        _pPixelBuffer->MakeCurrent();
		_DoSingleTimeStep();
		_bSingleStep = false;
	}
	
	if( _bSlow && (_iSlowDelay > 0) )
	{
		Sleep(_iSlowDelay);
	}
}


//.----------------------------------------------------------------------------.
//|   Function   : CA_Water::_DoSingleTime                                     |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void CA_Water::_DoSingleTimeStep()
{
    unsigned int iTemp;

  	// Swap texture source & target indices & pointers
	//  0 = start from initial loaded texture
	//  1/2 = flip flop back and forth between targets & sources

	switch( _iFlipState )
	{
    case 0:
        _iTexHeightInput    = _pDynamicTextureIDs[CA_TEXTURE_HEIGHT_SOURCE];       // initial height map.
        _iTexHeightOutput   = _pDynamicTextureIDs[CA_TEXTURE_HEIGHT_TARGET];    // next height map.

        _iTexVelocityInput  = _pDynamicTextureIDs[CA_TEXTURE_VELOCITY_SOURCE];  // initial velocity.
        _iTexVelocityOutput = _pDynamicTextureIDs[CA_TEXTURE_VELOCITY_TARGET];  // next velocity.

        // Clear initial velocity texture to 0x80 == gray
        glClearColor(0.5f, 0.5f, 0.5f, 1.0);
        glClear(GL_COLOR_BUFFER_BIT);

        // Now we need to copy the resulting pixels into the intermediate force field texture
        glActiveTextureARB(GL_TEXTURE0_ARB);
        glBindTexture(GL_TEXTURE_2D, _iTexVelocityInput);

        // use CopyTexSubImage for speed (even though we copy all of it) since we pre-allocated the texture
        glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, _pInitialMapDimensions[0], _pInitialMapDimensions[1]);

        break;  
        
    case 1:
		iTemp               = _iTexHeightInput;
        _iTexHeightInput    = _iTexHeightOutput;
        _iTexHeightOutput   = iTemp;

        iTemp               = _iTexVelocityInput;
        _iTexVelocityInput  = _iTexVelocityOutput;
        _iTexVelocityOutput = iTemp;

		break;

    case 2:
        iTemp               = _iTexHeightInput;
        _iTexHeightInput    = _iTexHeightOutput;
        _iTexHeightOutput   = iTemp;

        iTemp               = _iTexVelocityInput;
        _iTexVelocityInput  = _iTexVelocityOutput;
        _iTexVelocityOutput = iTemp;
        break;
	}
	
	// even if wireframe mode, render to texture as solid
	glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
	
    /////////////////////////////////////////////////////////////
	//  Render first 3 components of force from three neighbors
	//  Offsets selected are 1 center texel for center height
	//    and 3 of the 4 nearest neighbors.  Texture selected
	//    is same for all stages as we're turning height difference
	//    of nearest neightbor texels into a force value.

    glCallList(_pDisplayListIDs[CA_REGCOMBINER_NEIGHBOR_FORCE_CALC_1]);

    // set current source texture for stage 0 texture
    for (int i = 0; i < 4; i++)
    {
        glActiveTextureARB(GL_TEXTURE0_ARB + i);
        glBindTexture(GL_TEXTURE_2D, _iTexHeightInput);
        glEnable(GL_TEXTURE_2D);
    }

    GLenum wrapMode = _bWrap ? GL_REPEAT : GL_CLAMP_TO_EDGE;
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, wrapMode);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, wrapMode);

    // disable blending
    glDisable(GL_BLEND);

    // render using offset 1 (type 1 -- center + 3 of 4 nearest neighbors).
    glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, CV_UV_OFFSET_TO_USE, 1, 0, 0, 0);

    // bind the vertex program to be used for this step and the next one.
    glBindProgramNV(GL_VERTEX_PROGRAM_NV, _iVertexProgramID);
    glEnable(GL_VERTEX_PROGRAM_NV);

    // render a screen quad. with texture coords doing difference of nearby texels for force calc.
    glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);

    // Now we need to copy the resulting pixels into the intermediate force field texture
    glActiveTextureARB(GL_TEXTURE2_ARB);
    glBindTexture(GL_TEXTURE_2D, _pDynamicTextureIDs[CA_TEXTURE_FORCE_INTERMEDIATE]);

    // use CopyTexSubImage for speed (even though we copy all of it) since we pre-allocated the texture
    glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, _pInitialMapDimensions[0], _pInitialMapDimensions[1]);

    ////////////////////////////////////////////////////////////////
	// Now add in last component of force for the 4th neighbor
	//  that we didn't have enough texture lookups to do in the 
	//  first pass

    glCallList(_pDisplayListIDs[CA_REGCOMBINER_NEIGHBOR_FORCE_CALC_2]);
    
    // Cannot use additive blending as the force contribution might
	//   be negative and would have to subtract from the dest.
	// We must instead use an additional texture as target and read
	//   the previous partial 3-neighbor result into the pixel shader
	//   for possible subtraction

	// Alphablend must be false

	//; t0 = center  (same as last phase)
	//; t1 = 2nd axis final point (same as last phase)
	//; t2 = previous partial result texture sampled at center (result of last phase copied to texture)
	//; t3 = not used (disable now)

    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, wrapMode);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, wrapMode);

    glActiveTextureARB(GL_TEXTURE3_ARB);
    glDisable(GL_TEXTURE_2D);

    // vertex program already bound.
    // render using offset 2 (type 2 -- final nearest neighbor plus center of previous result).
    glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, CV_UV_OFFSET_TO_USE, 2, 0, 0, 0);

    // render a screen quad
    glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);

    // Now we need to copy the resulting pixels into the intermediate force field texture
    glActiveTextureARB(GL_TEXTURE1_ARB);
    glBindTexture(GL_TEXTURE_2D, _pDynamicTextureIDs[CA_TEXTURE_FORCE_TARGET]);

    // use CopyTexSubImage for speed (even though we copy all of it) since we pre-allocated the texture
    glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, _pInitialMapDimensions[0], _pInitialMapDimensions[1]);

	/////////////////////////////////////////////////////////////////
	// Apply the force with a scale factor to reduce it's magnitude.
	// Add this to the current texture representing the water height.
    
    glCallList(_pDisplayListIDs[CA_REGCOMBINER_APPLY_FORCE]);

    // use offsets of zero
    glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, CV_UV_OFFSET_TO_USE, 0, 0, 0, 0);

    // bind the vertex program to be used for this step and the next one.

    glActiveTextureARB(GL_TEXTURE0_ARB);
    glBindTexture(GL_TEXTURE_2D, _iTexVelocityInput);
    glActiveTextureARB(GL_TEXTURE1_ARB);
    glBindTexture(GL_TEXTURE_2D, _pDynamicTextureIDs[CA_TEXTURE_FORCE_TARGET]);
    glActiveTextureARB(GL_TEXTURE2_ARB);
    glDisable(GL_TEXTURE_2D);
    glActiveTextureARB(GL_TEXTURE3_ARB);
    glDisable(GL_TEXTURE_2D);

    // Draw the quad to add in force.
    glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);

    ///////////////////////////////////////////////////////////////////
	// With velocity texture selected, render new excitation droplets
	//   at random freq.

	float rRandomFrequency;
	rRandomFrequency = (float)rand()/((float)RAND_MAX);

	if( _rDropletFrequency > rRandomFrequency)
	{
		// a drop falls - decide where
        Droplet drop;

		drop.rX = 2 * ((float)rand()/((float)RAND_MAX) - 0.5f);
		drop.rY = 2 * ((float)rand()/((float)RAND_MAX) - 0.5f);

		// pick a droplet size at random:
		drop.rScale = 0.02f +  0.1f * ((float)rand()/((float)RAND_MAX));

		AddDroplet( drop );
	}

	//  Now draw the droplets:
	if (_droplets.size() > 0)
    {
       _DrawDroplets();
       _droplets.clear();
    }

    // Now we need to copy the resulting pixels into the velocity texture
    glActiveTextureARB(GL_TEXTURE1_ARB);
    glBindTexture(GL_TEXTURE_2D, _iTexVelocityOutput);

    // use CopyTexSubImage for speed (even though we copy all of it) since we pre-allocated the texture
    glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, _pInitialMapDimensions[0], _pInitialMapDimensions[1]);  

    //////////////////////////////////////////////////////////////////////
	// Apply velocity to position
    glCallList(_pDisplayListIDs[CA_REGCOMBINER_APPLY_VELOCITY]);
    glEnable(GL_VERTEX_PROGRAM_NV);

    glActiveTextureARB(GL_TEXTURE0_ARB);
    glBindTexture(GL_TEXTURE_2D, _iTexHeightInput);
    glActiveTextureARB(GL_TEXTURE1_ARB); // velocity output already bound
    glEnable(GL_TEXTURE_2D);

    // use offsets of zero
    glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, CV_UV_OFFSET_TO_USE, 0, 0, 0, 0);

    // Draw the quad to add in force.
    glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);

    // Now we need to copy the resulting pixels into the input height texture
    glActiveTextureARB(GL_TEXTURE0_ARB);
    glBindTexture(GL_TEXTURE_2D, _iTexHeightInput);
    
    // use CopyTexSubImage for speed (even though we copy all of it) since we pre-allocated the texture
    glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, _pInitialMapDimensions[0], _pInitialMapDimensions[1]); 

    ///////////////////////////////////////////////////////////////////
	//  blur positions to smooth noise & generaly dampen things
	//  degree of blur is controlled by magnitude of 4 neighbor texel
	//   offsets with bilinear on
    
    for (i = 1; i < 4; i++)
    {
        glActiveTextureARB(GL_TEXTURE0_ARB + i);
        glBindTexture(GL_TEXTURE_2D, _iTexHeightInput);
        glEnable(GL_TEXTURE_2D);
    }

    // use offsets of 3
    glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, CV_UV_OFFSET_TO_USE, 3, 0, 0, 0);

    glCallList(_pDisplayListIDs[CA_REGCOMBINER_EQ_WEIGHT_COMBINE]);

    glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);

    // Draw the logo in the water.
    if( _bApplyInteriorBoundaries )
	{
        glDisable(GL_VERTEX_PROGRAM_NV);
		_DrawInteriorBoundaryObjects();
	}

    // Now we need to copy the resulting pixels into the velocity texture
    glActiveTextureARB(GL_TEXTURE0_ARB);
    glBindTexture(GL_TEXTURE_2D, _iTexHeightOutput);

    
    // use CopyTexSubImage for speed (even though we copy all of it) since we pre-allocated the texture
    glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, _pInitialMapDimensions[0], _pInitialMapDimensions[1]);
 
    ///////////////////////////////////////////////////////////////////
	// If selected, create a normal map from the height
	
	if( _bCreateNormalMap )
	{
		_CreateNormalMap();
	}
	
    ///////////////////////////////////////////////////////////
	// Flip the state variable for the next round of rendering
	switch( _iFlipState )
	{
	case 0:
		_iFlipState = 1;
		break;
	case 1:
		_iFlipState = 2;
		break;
	case 2:
		_iFlipState = 1;
		break;
	}
}



//.----------------------------------------------------------------------------.
//|   Function   : CA_Water::_CreateNormalMap                                  |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void CA_Water::_CreateNormalMap()
{
    // use the height output on all four texture stages
    for (int i = 0; i < 4; i++)
    {
        glActiveTextureARB(GL_TEXTURE0_ARB + i);
        glBindTexture(GL_TEXTURE_2D, _iTexHeightOutput);
        glEnable(GL_TEXTURE_2D);
    }

    // Set constants for red & green scale factors (also essential color masks)
    // Red mask first
    float pPixMasks[4] = { _rNormalSTScale, 0.0f, 0.0f, 0.0f };
    glCombinerStageParameterfvNV(GL_COMBINER2_NV, GL_CONSTANT_COLOR0_NV, pPixMasks);

    // Now green mask & scale:
	pPixMasks[0] = 0.0f;
	pPixMasks[1] = _rNormalSTScale;
    glCombinerStageParameterfvNV(GL_COMBINER2_NV, GL_CONSTANT_COLOR1_NV, pPixMasks);

    glCallList(_pDisplayListIDs[CA_REGCOMBINER_CREATE_NORMAL_MAP]);

    // set vp offsets to nearest neighbors
    glProgramParameter4fNV(GL_VERTEX_PROGRAM_NV, CV_UV_OFFSET_TO_USE, 4, 0, 0, 0);
    glEnable(GL_VERTEX_PROGRAM_NV);
    
    glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);

    // Now we need to copy the resulting pixels into the normal map
    glActiveTextureARB(GL_TEXTURE0_ARB);
    glBindTexture(GL_TEXTURE_2D, _pDynamicTextureIDs[CA_TEXTURE_NORMAL_MAP]);
    
    // use CopyTexSubImage for speed (even though we copy all of it) since we pre-allocated the texture
    glCopyTexSubImage2D(GL_TEXTURE_2D, 0, 0, 0, 0, 0, _pInitialMapDimensions[0], _pInitialMapDimensions[1]);
}

//.----------------------------------------------------------------------------.
//|   Function   : CA_Water::_DrawInteriorBoundaryObjects                      |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void CA_Water::_DrawInteriorBoundaryObjects()
{
    glDisable(GL_REGISTER_COMBINERS_NV);
    
    glActiveTextureARB(GL_TEXTURE0_ARB);
    glBindTexture(GL_TEXTURE_2D, _pStaticTextureIDs[CA_TEXTURE_INITIAL_MAP]);
    glEnable(GL_TEXTURE_2D);

    glEnable(GL_ALPHA_TEST);

    // disable other texture units.
    for (int i = 1; i < 4; i++)
    {
        glActiveTextureARB(GL_TEXTURE0_ARB + i);
        glDisable(GL_TEXTURE_2D);
    }
    
    glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
    glEnable(GL_BLEND);

    glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);

    if (_bSpinLogo)
    {
        glActiveTextureARB(GL_TEXTURE0_ARB);
        glBindTexture(GL_TEXTURE_2D, _pStaticTextureIDs[CA_TEXTURE_SPIN]);
        static int angle = 0;
        glMatrixMode(GL_MODELVIEW);
        glPushMatrix();
        glRotatef(angle, 0, 0, 1);
        angle += 1;

        glCallList(_pDisplayListIDs[CA_DRAW_SCREEN_QUAD]);

        glPopMatrix();
    }

    glDisable(GL_ALPHA_TEST);
    glDisable(GL_BLEND);
}


//.----------------------------------------------------------------------------.
//|   Function   : CreateTextureObject                                         |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void CreateTextureObject(unsigned int ID, unsigned int iWidth, unsigned int iHeight, unsigned char *pData)
{
    glBindTexture(  GL_TEXTURE_2D, ID);
    glTexImage2D (  GL_TEXTURE_2D, 
                    0, 
                    GL_RGBA8, 
                    iWidth,
                    iHeight, 
                    0, 
                    GL_BGRA_EXT,
                    GL_UNSIGNED_BYTE, 
                    pData);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);
}


//.----------------------------------------------------------------------------.
//|   Function   : LoadCubeMap                                                 |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void LoadCubeMap(unsigned int ID, const char *pFilename, bool bMipmap)
{
    GLenum faces [] = { GL_TEXTURE_CUBE_MAP_POSITIVE_X_ARB,
                        GL_TEXTURE_CUBE_MAP_NEGATIVE_X_ARB,
	                    GL_TEXTURE_CUBE_MAP_POSITIVE_Y_ARB,
	                    GL_TEXTURE_CUBE_MAP_NEGATIVE_Y_ARB,
	                    GL_TEXTURE_CUBE_MAP_POSITIVE_Z_ARB,
                        GL_TEXTURE_CUBE_MAP_NEGATIVE_Z_ARB };
    char *faceNames[] = {"posx", "negx", "posy", "negy", "posz", "negz" };

    // create and bind a cubemap texture object
    glBindTexture(GL_TEXTURE_CUBE_MAP_ARB, ID);

    // enable automipmap generation if needed.
    glTexParameteri(GL_TEXTURE_CUBE_MAP_ARB, GL_GENERATE_MIPMAP_SGIS, bMipmap);
    
    if (bMipmap)
        glTexParameterf(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_MIN_FILTER, GL_LINEAR_MIPMAP_LINEAR);
    else
        glTexParameterf(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
    glTexParameterf(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
    glTexParameterf(GL_TEXTURE_CUBE_MAP_ARB, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

    // load 6 faces.
    for (int i = 0; i < 6; i++)
    {
        char buffer[FILENAME_MAX];
        sprintf(buffer, pFilename, faceNames[i]);
        tga::tgaImage *pImage = tga::read(buffer);

        if (!pImage)
        {    
            fprintf(stderr, "Could not load image.   Exiting. %s\n", buffer);
            exit(1);
        }

        
        glTexImage2D(faces[i], 
                     0, 
                     GL_RGBA8, 
                     pImage->width, 
                     pImage->height, 
                     0, 
                     pImage->format, 
                     GL_UNSIGNED_BYTE, 
                     pImage->pixels);
    }
}


//.----------------------------------------------------------------------------.
//|   Function   : CA_Water::LoadTextures                                      |
//|   Description: Load the textures used to run the CA.                       |
//.----------------------------------------------------------------------------.
void CA_Water::_LoadTextures(char *pInitialMapFilename, char *pSpinFilename, char *pDropletFilename, char *pCubeMapFilename)
{
    // load the starting texture
    tga::tgaImage *pImage   = tga::read(pInitialMapFilename);
    if (!pImage)
    {
        fprintf(stderr, "Could not load image.   Exiting. %s\n", pInitialMapFilename);
        exit(1);
    }

    tga::tgaImage *pSpin = tga::read(pSpinFilename);
    if (!pImage)
    {
        fprintf(stderr, "Could not load image.   Exiting. %s\n", pInitialMapFilename);
        exit(1);
    }

    tga::tgaImage *pDroplet = tga::read(pDropletFilename);
    if (!pDroplet)
    {
        fprintf(stderr, "Could not load image.   Exiting. %s\n", pInitialMapFilename);
        exit(1);
    }
    
    _pInitialMapDimensions[0] = pImage->width;
    _pInitialMapDimensions[1] = pImage->height;

    glGenTextures(CA_NUM_STATIC_TEXTURES, _pStaticTextureIDs); 
    glGenTextures(CA_NUM_DYNAMIC_TEXTURES, _pDynamicTextureIDs); // also create intermediate texture object

    // upload the initial map texture
    CreateTextureObject(_pStaticTextureIDs[CA_TEXTURE_INITIAL_MAP],
                        _pInitialMapDimensions[0], 
                        _pInitialMapDimensions[1], 
                        pImage->pixels);

    CreateTextureObject(_pStaticTextureIDs[CA_TEXTURE_SPIN],
                        _pInitialMapDimensions[0], 
                        _pInitialMapDimensions[1], 
                        pSpin->pixels);

    CreateTextureObject(_pStaticTextureIDs[CA_TEXTURE_DROPLET], 
                        pDroplet->width, 
                        pDroplet->height, 
                        pDroplet->pixels);

    // load the cubemap texture
    LoadCubeMap(_pStaticTextureIDs[CA_TEXTURE_CUBEMAP], pCubeMapFilename, true);

    for (int i = 0; i < CA_NUM_DYNAMIC_TEXTURES; i++)
    {
        // now create a dummy intermediate textures from the initial map texture
        CreateTextureObject(_pDynamicTextureIDs[i],
                            _pInitialMapDimensions[0], 
                            _pInitialMapDimensions[1], 
                            pImage->pixels);
    }

    delete pImage;

    _iTexHeightInput    = _pStaticTextureIDs[CA_TEXTURE_INITIAL_MAP];       // initial height map.
    _iTexHeightOutput   = _pDynamicTextureIDs[CA_TEXTURE_HEIGHT_TARGET];    // next height map.
    
    _iTexVelocityInput  = _pDynamicTextureIDs[CA_TEXTURE_VELOCITY_SOURCE];  // initial velocity.
    _iTexVelocityOutput = _pDynamicTextureIDs[CA_TEXTURE_VELOCITY_TARGET];  // next velocity.
}


//.----------------------------------------------------------------------------.
//|   Function   : CA_Water::_CreateAndWriteUVOffsets                          |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void CA_Water::_CreateAndWriteUVOffsets(unsigned int width, unsigned int height)
{
    // This sets vertex shader constants used to displace the
	//  source texture over several additive samples.  This is 
	//  used to accumulate neighboring texel information that we
	//  need to run the game - the 8 surrounding texels, and the 
	//  single source texel which will either spawn or die in the 
	//  next generation.
	// Label the texels as follows, for a source texel "e" that
	//  we want to compute for the next generation:
	//		
    //          abc
    //          def
    //          ghi:

    // first the easy one: no offsets for sampling center
	//  occupied or unoccupied
	// Use index offset value 0.0 to access these in the 
	//  vertex shader.
    
    _rPerTexelWidth  = 1.0f / static_cast<float>(width);
    _rPerTexelHeight = 1.0f / static_cast<float>(height);

  	// Offset set 0 : center texel sampling
    float noOffsetX[4] = { 0, 0, 0, 0 };
    float noOffsetY[4] = { 0, 0, 0, 0 };

    // Offset set 1:  For use with neighbor force pixel shader 1
	//  samples center with 0, +u, -u, and +v,
	//	ie the 'e','d', 'f', and 'h' texels

    float rDist = 1.5f;

    float type1OffsetX[4] = { 0.0f, -rDist * _rPerTexelWidth, rDist * _rPerTexelWidth, rDist * _rPerTexelWidth      };
    float type1OffsetY[4] = { 0.0f, rDist * _rPerTexelHeight, rDist * _rPerTexelHeight, -rDist * _rPerTexelHeight   };


    // Offset set 2:  for use with neighbor force pixel shader 2
	//  samples center with 0, and -v texels 
	//  ie the 'e' and 'b' texels
	// This completes a pattern of sampling center texel and it's
	//   4 nearest neighbors to run the height-based water simulation
	// 3rd must be 0 0 to sample texel center from partial result
	//   texture.

    float type2OffsetX[4] = { 0.0f, -rDist * _rPerTexelWidth, 0.0f, 0.0f    };
    float type2OffsetY[4] = { 0.0f, -rDist * _rPerTexelHeight, 0.0f, 0.0f   };
        
    // type 3 offsets
  	_UpdateBlurVertOffset();

    /////////////////////////////////////////////////////////////
    // Nearest neighbor offsets:

   	float type4OffsetX[4] = { -_rPerTexelWidth, _rPerTexelWidth, 0.0f,              0.0f   };
    float type4OffsetY[4] = { 0.0f,             0.0f,            -_rPerTexelHeight, _rPerTexelHeight };

    // write all these offsets to constant memory
    for (int i = 0; i < 4; ++i)
    {
        float noOffset[]    = { noOffsetX[i],    noOffsetY[i],    0.0f, 0.0f };
        float type1Offset[] = { type1OffsetX[i], type1OffsetY[i], 0.0f, 0.0f };
        float type2Offset[] = { type2OffsetX[i], type2OffsetY[i], 0.0f, 0.0f };
        float type4Offset[] = { type4OffsetX[i], type4OffsetY[i], 0.0f, 0.0f };

        glProgramParameter4fvNV(GL_VERTEX_PROGRAM_NV, CV_UV_T0_NO_OFFSET + 5 * i, noOffset);
        glProgramParameter4fvNV(GL_VERTEX_PROGRAM_NV, CV_UV_T0_TYPE1     + 5 * i, type1Offset);
        glProgramParameter4fvNV(GL_VERTEX_PROGRAM_NV, CV_UV_T0_TYPE2     + 5 * i, type2Offset);
        glProgramParameter4fvNV(GL_VERTEX_PROGRAM_NV, CV_UV_T0_TYPE4     + 5 * i, type4Offset);
    }
}


//.----------------------------------------------------------------------------.
//|   Function   : CA_Water::_UpdateBlurVertOffset                             |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void CA_Water::_UpdateBlurVertOffset()
{
    _pPixelBuffer->MakeCurrent();

    static float type3OffsetX[4] = {   -_rPerTexelWidth * 0.5f, 
                                        _rPerTexelWidth, 
                                        _rPerTexelWidth * 0.5f, 
                                       -_rPerTexelWidth 
                                    };
	
    static float type3OffsetY[4] = {    _rPerTexelHeight,
							            _rPerTexelHeight * 0.5f,
							           -_rPerTexelHeight,
							           -_rPerTexelHeight * 0.5f 
                                    };

    float offsets[4] = { 0, 0, 0, 0 };


    for (int i = 0; i < 4; ++i)
    {
		offsets[0] = _rBlurDist * ( type3OffsetX[i]);
		offsets[1] = _rBlurDist * ( type3OffsetY[i]);
    	glProgramParameter4fvNV(  GL_VERTEX_PROGRAM_NV, CV_UV_T0_TYPE3 + 5 * i, offsets);
    }
}


//.----------------------------------------------------------------------------.
//|   Function   : CA_Water::AddDroplet                                       |
//|   Description:                                                             |
//.----------------------------------------------------------------------------.
void CA_Water::AddDroplet( const Droplet &drop )
{
	// Adds a droplet to the rendering queue.
	// These are then rendered when DrawDroplets() is called
	// Coords are from 0 to 1.0 across texture surface
	// Size of droplet is determined at draw time - Could add 
	//   another variable to override this
	_droplets.push_back( drop );
}

void CA_Water::_DrawDroplets()
{
    glDisable(GL_REGISTER_COMBINERS_NV);
    glDisable(GL_VERTEX_PROGRAM_NV);

    glActiveTextureARB(GL_TEXTURE0_ARB);
    glBindTexture(GL_TEXTURE_2D, _pStaticTextureIDs[CA_TEXTURE_DROPLET]);
    glEnable(GL_TEXTURE_2D);

    glActiveTextureARB(GL_TEXTURE1_ARB);
    glDisable(GL_TEXTURE_2D);

    glBlendFunc(GL_ONE, GL_ONE);
    glEnable(GL_BLEND);

    glBegin(GL_QUADS);
    glColor4f(1, 1, 1, 1);
    for( int i=0; i < _droplets.size(); i++)
	{
        // coords in [-1,1] range

        // Draw a single quad to the texture render target
		// The quad is textured with the initial droplet texture, and
		//   covers some small portion of the render target
		// Draw the droplet
       
        glTexCoord2f(0, 0); glVertex2f(_droplets[i].rX - _droplets[i].rScale,  _droplets[i].rY - _droplets[i].rScale);
        glTexCoord2f(1, 0); glVertex2f(_droplets[i].rX + _droplets[i].rScale,  _droplets[i].rY - _droplets[i].rScale);
        glTexCoord2f(1, 1); glVertex2f(_droplets[i].rX + _droplets[i].rScale,  _droplets[i].rY + _droplets[i].rScale);
        glTexCoord2f(0, 1); glVertex2f(_droplets[i].rX - _droplets[i].rScale,  _droplets[i].rY + _droplets[i].rScale);          
    }
    glEnd();

    glDisable(GL_BLEND);
}
